"use strict";

{
	// In the C3 runtime's worker mode, all the runtime scripts (e.g. plugin.js, instance.js, actions.js)
	// are loaded in a Web Worker, which has no access to the document so cannot make DOM calls. To help
	// plugins use DOM elements the runtime internally manages a postMessage() bridge wrapped in some helper
	// classes designed to manage DOM elements. Then this script (domSide.js) is loaded in the main document
	// (aka the main thread) where it can make any DOM calls on behalf of the runtime. Conceptually the two
	// ends of the messaging bridge are the "Runtime side" in a Web Worker, and the "DOM side" with access
	// to the Document Object Model (DOM). The addon's plugin.js specifies to load this script on the
	// DOM side by making the call: this._info.SetDOMSideScripts(["c3runtime/domSide.js"])
	// Note that when NOT in worker mode, this entire framework is still used identically, just with both
	// the runtime and the DOM side in the main thread. This allows non-worker mode to work the same with
	// no additional code changes necessary. However it's best to imagine that the runtime side is in a
	// Web Worker, since that is when it is necessary to separate DOM calls from the runtime.
	
	// NOTE: use a unique DOM component ID to ensure it doesn't clash with anything else
	// This must also match the ID in instance.js and plugin.js.
	const DOM_COMPONENT_ID = "ppstudio_handy_DOM";

	const HANDLER_CLASS = class ppstudio_handy_utilities_DOMHandler extends DOMElementHandler
	{
		constructor(iRuntime)
		{
			super(iRuntime, DOM_COMPONENT_ID);
			this._GA = null;
			this._GAEnabled = false;
			
			//Listeners to the DOM events must be registered
			this.AddRuntimeMessageHandlers([
					["load", d => this._ActivateAnalytics(d)],
					["send-event", d => this._SendEvent(d)],
					["inject-css", d => this._CreateCSSData(d)]
				]
			);
		}
		
		_ActivateAnalytics(data){
			if (!window["ga"])
				return;
			
			if (window["ga"]!=="undefined"){
				window["ga"]('create', data["gaID"], 'auto');
				window["ga"]('send', 'pageview');
				this._GA = window["ga"];
				this._GAEnabled = true;
			}
			
			return {
				"GAEnabled": this._GAEnabled
			}
		}
		
		_SendEvent(data){
			this._GA('send','event',data["category"],data["activity"],data["label"],data["value"]);
			
			return {}; //An event must always return an object. It can be empty
		}
		
		_CreateCSSData(data){
			if (data){
				var styleTag = document.createElement("style");
				styleTag.innerHTML =  data["css-code"];
				document.getElementsByTagName("head")[0].appendChild(styleTag);
			}
		}
	};
	
	RuntimeInterface.AddDOMHandlerClass(HANDLER_CLASS);
}