
const C3 = globalThis.C3;

C3.Plugins.Rex_LayerObj.Instance = class Rex_LayerObjInstance extends globalThis.ISDKWorldInstanceBase
{
	constructor()
	{
	    super();

	    // Initialize from editor properties
	    const properties = this._getInitProperties();
	    this._controlledVisibility = properties?.[0] ?? true;
	    if (typeof this._controlledVisibility !== "boolean") {
	        this._controlledVisibility = this._controlledVisibility !== 0;
	    }
	    
	    this.isVisible = this._controlledVisibility;

	    // Dimension tracking
	    this.widthInit = null;
	    this.heightInit = null;

	    // Layer caching
	    this._cachedLayer = null;
	    this._lastLayerIndex = -1;

	    // State tracking for change detection
	    this._lastState = {
	        scale: -1,
	        angle: -1,
	        opacity: -1,
	        isVisible: null
	    };

	    this._initialStateApplied = false;
	    this._setTicking2(true);
	}

	_onCreate()
	{
	    this._initializeDimensions();
	}

	_release()
	{
	    this._setTicking2(false);
	    this._cachedLayer = null;
	    super._release();
	}

	_saveToJson()
	{
	    return {
	        wi: this.widthInit,
	        hi: this.heightInit,
	        cv: this._controlledVisibility
	    };
	}

	_loadFromJson(o)
	{
	    this.widthInit = o.wi;
	    this.heightInit = o.hi;
	    this._controlledVisibility = o.cv ?? true;
	}

	onPropertyChanged(id)
	{
	    if (id === "visible") {
	        this._controlledVisibility = this.isVisible;
	    }
	}

	_initializeDimensions()
	{
	    if (this.width > 0 && this.height > 0 && !this.widthInit) {
	        this.widthInit = this.width;
	        this.heightInit = this.height;
	    }
	}

	_getTargetLayer()
	{
	    const layout = this.runtime?.layout;
	    if (!layout) return null;

	    const targetLayerIndex = this.layer?.index ?? this.layerIndex ?? 1;

	    if (this._cachedLayer && this._lastLayerIndex === targetLayerIndex) {
	        return this._cachedLayer;
	    }

	    this._cachedLayer = layout.getLayer(targetLayerIndex);
	    this._lastLayerIndex = targetLayerIndex;
	    return this._cachedLayer;
	}

	_hasChanged(current, previous, threshold = 0.001)
	{
	    return Math.abs(current - previous) > threshold;
	}

	_calculateCurrentScale()
	{
	    if (!this.widthInit || !this.heightInit) return 1;
	    
	    const wScale = this.width / this.widthInit;
	    const hScale = this.height / this.heightInit;
	    return Math.min(wScale, hScale);
	}

	_applyLayerState(layer, scale, angle, opacity, isVisible)
	{
	    layer.scale = scale;
	    layer.angle = angle;
	    layer.opacity = opacity;
	    layer.isVisible = isVisible;

	    // Update cached state
	    Object.assign(this._lastState, { scale, angle, opacity, isVisible });
	}

	_tick2()
	{
	    const layer = this._getTargetLayer();
	    if (!layer) return;

	    this._initializeDimensions();
	    if (!this.widthInit || !this.heightInit) return;

	    // Calculate current values
	    const scale = this._calculateCurrentScale();
	    const angle = this.angle ?? 0;
	    const opacity = C3.clamp(this.opacity ?? 1, 0, 1);
	    const isVisible = this._controlledVisibility;

	    // Apply initial state on first tick
	    if (!this._initialStateApplied) {
	        this._applyLayerState(layer, scale, angle, opacity, isVisible);
	        this._initialStateApplied = true;
	        return;
	    }

	    // Sync visibility from runtime changes
	    if (this.isVisible !== this._controlledVisibility) {
	        this._controlledVisibility = this.isVisible;
	    }

	    // Batch updates - only apply changes when needed
	    let hasChanges = false;

	    if (this._hasChanged(scale, this._lastState.scale)) {
	        layer.scale = scale;
	        this._lastState.scale = scale;
	        hasChanges = true;
	    }

	    if (this._hasChanged(angle, this._lastState.angle)) {
	        layer.angle = angle;
	        this._lastState.angle = angle;
	        hasChanges = true;
	    }

	    if (this._hasChanged(opacity, this._lastState.opacity)) {
	        layer.opacity = opacity;
	        this._lastState.opacity = opacity;
	        hasChanges = true;
	    }

	    if (isVisible !== this._lastState.isVisible) {
	        layer.isVisible = isVisible;
	        this._lastState.isVisible = isVisible;
	        hasChanges = true;
	    }

	    // Single log for all changes (optional - remove for production)
	    if (hasChanges) {
	        
	    }
	}

	_draw() {} // Required by SDK

};
